// SPDX-License-Identifier: BSD-2-Clause
/*
 * Copyright (C) 2016 The Android Open Source Project
 */
/* Portions Copyright (C) 2019-2023 Amazon.com, Inc. or its affiliates. All Rights Reserved. */

#include <common.h>
#include <fastboot.h>
#include <fastboot-internal.h>
#include <stdlib.h>

#ifdef CONFIG_FASTBOOT_FLASH_MMC
#include <fb_mmc.h>
#elif defined(CONFIG_FASTBOOT_FLASH_NAND)
#include <fb_nand.h>
#elif defined(CONFIG_FASTBOOT_FLASH_CUS)
#include <fb_cus.h>
#endif
/*
COCOA-768: Add USB fastboot to uBoot
COCOA-782: Add OEM CODA support
COCOA-2128: Implement fastboot lockdown
*/
#include <unlocker.h>
#ifdef CONFIG_FASTBOOT_CMD_OEM
#include <ubi_uboot.h>
#include <drvGPIO.h>
#include <ring_gpio_config.h>
#endif

#ifdef CONFIG_CMD_OTPCTRL
#define RSA_KEY_SIZE 256
#define AES_KEY_SIZE 16  //bytes
#ifndef CID
#error No CID
#endif
#define OTP_CELL_SIZE 4
#define RETRY_THRESHOLD 2
#define BOOT_SRC 0x04
#define E_KEY 0x01000100
#define E_KEY_SIZE OTP_CELL_SIZE
#define MAX_ERR_MSG_LENGTH 25
#define MAX_DATA_MSG_LENGTH 12
#define OTP_PREFIX "otp_"
#define MAX_KEY_DATA_SIZE 256
#endif
/**
 * image_size - final fastboot image size
 */
static u32 image_size;

/**
 * fastboot_bytes_received - number of bytes received in the current download
 */
static u32 fastboot_bytes_received;

/**
 * fastboot_bytes_expected - number of bytes expected in the current download
 */
static u32 fastboot_bytes_expected;

static void okay(char *, char *);
static void unrecognized(char *, char *);
static void getvar(char *, char *);
static void download(char *, char *);
#ifdef CONFIG_FASTBOOT_FLASH
static void flash(char *, char *);
static void erase(char *, char *);
#endif
static void reboot_bootloader(char *, char *);
#ifdef CONFIG_FASTBOOT_CMD_OEM
static void oem_ubi(char *, char *);
static void oem_coda(char *, char *);
static void oem_fbtool(char *, char *);
static void oem_setserial(char *, char *);
static void oem_setvar(char *, char *);
static void oem_relock(char *, char *);
static void oem_setfactory(char *, char *);
#ifdef CONFIG_CMD_OTPCTRL
static void oem_otp_w(char *cmd_parameter, char *response);
static void oem_otp_r(unsigned long otp_cmd, unsigned long dataSize, char *response);
static bool oem_otp_strcmp_funct(char *cmd_parameter,
                                 unsigned long* otp_cmd,
                                 unsigned long* val,
                                 unsigned long* dataSize,
                                 u8** defaultData);
#endif
#ifdef CONFIG_FASTBOOT_FLASH_NAND
static void oem_otp_nand_lock(char *cmd_parameter, char *response);
#endif
static void oem_ubi_validate(char *cmd_parameter, char *response);
static void handle_oem_command(char *, char *);
#endif

#ifdef CONFIG_FASTBOOT_FLASH_NAND
extern int writeSpinandCIS(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[]);
extern int write_unlock_signature(uint8_t *unlock_signature, int unlock_sig_len);
#endif


struct cmd_entry {
	const char *command;
	void (*dispatch)(char *cmd_parameter, char *response);
};

static inline bool is_safe_to_write(void *base_addr, u32 len);

static int handle_commands_internal(const struct cmd_entry *cmds,
                                    int num_commands,
                                    char *cmd_string,
                                    char *response);

#define FB_ALLOWLIST_BASE	5	// getvar, boot, continue, reboot,
	                                // reboot-bootloader

#ifdef CONFIG_FASTBOOT_CMD_OEM
#define FB_ALLOWLIST_OEM	1
#else
#define FB_ALLOWLIST_OEM	0
#endif

#ifdef CONFIG_FASTBOOT_FLASH
#define FB_ALLOWLIST_FLASH	2	// flash, download (unlock handled in func flash requires both)
#else
#define FB_ALLOWLIST_FLASH	0
#endif

#define FB_ALLOWLIST (FB_ALLOWLIST_BASE + FB_ALLOWLIST_OEM + FB_ALLOWLIST_FLASH)
static const struct cmd_entry commands[] = {
	/* These are ordered according to the enum in fastboot.h.  They have
	   special handling elsewhere */
	[FASTBOOT_COMMAND_GETVAR] = {
		.command = "getvar",
		.dispatch = getvar
	},
	[FASTBOOT_COMMAND_BOOT] = {
		.command = "boot",
		.dispatch = unrecognized
	},
	[FASTBOOT_COMMAND_CONTINUE] = {
		.command = "continue",
		.dispatch = okay
	},
	[FASTBOOT_COMMAND_REBOOT] = {
		.command = "reboot",
		.dispatch = okay
	},
	[FASTBOOT_COMMAND_REBOOT_BOOTLOADER] = {
		.command = "reboot-bootloader",
		.dispatch = reboot_bootloader
	},
	/* Below here are in array order and are handled in this file */
#ifdef CONFIG_FASTBOOT_CMD_OEM
	{
		.command = "oem",
		.dispatch = handle_oem_command,
	},
#endif
#ifdef CONFIG_FASTBOOT_FLASH
	{
		.command = "flash",
		.dispatch = flash
	},
	{
		.command = "download",
		.dispatch = download
	},
	{
		.command = "erase",
		.dispatch = erase
	},
#endif
	{
		.command = "set_active",
		.dispatch = okay
	},
};

#ifdef CONFIG_FASTBOOT_CMD_OEM
#define OEM_ALLOWLIST	4	// Commands related to locking are allowed.
static const struct cmd_entry oem_commands[] = {
	{
		.command = "relock",
		.dispatch = oem_relock,
	},
	{
		.command = "coda",
		.dispatch = oem_coda
	},
	{
		.command = "otplock",
		.dispatch = oem_otp_nand_lock,
	},
	{
		.command = "ubival",
		.dispatch = oem_ubi_validate,
	},
	{
		.command = "fbtool",
		.dispatch = oem_fbtool
	},
	{
		.command = "setserial",
		.dispatch = oem_setserial
	},
	{
		.command = "setvar",
		.dispatch = oem_setvar
	},
	{
		.command = "ubi",
		.dispatch = oem_ubi,
	},

#ifdef CONFIG_CMD_OTPCTRL
	{
		/* OTP write operations */
		.command = "otpw",
		.dispatch = oem_otp_w,
	},
#endif
	{
		.command = "factorymode",
		.dispatch = oem_setfactory,
	}
};

static void handle_oem_command(char *cmd_string, char *response)
{
	int allowed_commands;
	unlock_state_t unlock_state;

	is_unlocked(&unlock_state);
	if (unlock_state < UNLOCK_STATE_RESTRICTED_UNLOCK)
		allowed_commands = OEM_ALLOWLIST;
	else
		allowed_commands = ARRAY_SIZE(oem_commands);

	if (handle_commands_internal(oem_commands, allowed_commands,
	                             cmd_string, response) < 0) {
		printf("ERROR: oem command %s not recognized.\n", cmd_string);
		fastboot_fail("unrecognized command", response);
	}
}
#endif

/**
 * fastboot_handle_command - Handle fastboot command
 *
 * @cmd_string: Pointer to command string
 * @response: Pointer to fastboot response buffer
 *
 * Return: Executed command, or -1 if not recognized
 */
int fastboot_handle_command(char *cmd_string, char *response)
{
	int ret, allowed_commands;
	unlock_state_t unlock_state;

	is_unlocked(&unlock_state);
	if (unlock_state < UNLOCK_STATE_RESTRICTED_UNLOCK)
		allowed_commands = FB_ALLOWLIST;
	else
		allowed_commands = ARRAY_SIZE(commands);

	ret = handle_commands_internal(commands,
	                                   allowed_commands,
	                                   cmd_string,
	                                   response);

	if (ret < 0) {
		printf("ERROR: command %s not recognized.\n", cmd_string);
		fastboot_fail("unrecognized command", response);
	}
	return ret;
}

static inline bool is_safe_to_write(void *base_addr, u32 len) {
#ifdef __GNUC__
	register uintptr_t sp asm("sp");
#else
	uintptr_t sp;
	sp = ((uintptr_t)&sp) - 0xC0;
#endif
	// back off 4K from where we think SP is pointing at
	// https://issues.labcollab.net/browse/COCOA-40796?focusedCommentId=41972956
	bool safe_to_write = (sp < base_addr) || ((base_addr + len) < (sp - 0x1000));
	return safe_to_write;
}

/*
int fastboot_handle_command(char *cmd_string, char *response)
{
	int i;
	char *cmd_parameter;

	cmd_parameter = cmd_string;
	strsep(&cmd_parameter, ":");

	for (i = 0; i < FASTBOOT_COMMAND_COUNT; i++) {
		if (!strcmp(commands[i].command, cmd_string)) {
			if (commands[i].dispatch) {
				commands[i].dispatch(cmd_parameter,
							response);
				return i;
			} else {
				break;
			}
		}
	}

	printf("ERR: command %s not recognized.\n", cmd_string);
	fastboot_fail("unrecognized command", response);
	return -1;
}
*/

static int handle_commands_internal(const struct cmd_entry *cmds,
                                    int num_commands,
                                    char *cmd_string,
                                    char *response)
{
	int i;
	char *cmd_param;

	for (i = 0; i < num_commands; i++) {
		if (!strncmp(cmds[i].command, cmd_string, strlen(cmds[i].command))) {
			if (cmds[i].dispatch) {
				cmd_param = cmd_string + strlen(cmds[i].command);
				/* handle space or : separators */
				if (cmd_param[0] == ':' || cmd_param[0] == ' ')
					cmd_param++;
				cmds[i].dispatch(cmd_param, response);
				return i;
			} else {
				break;
			}
		}
	}
	return -1;
}


/**
 * okay() - Send bare OKAY response
 *
 * @cmd_parameter: Pointer to command parameter
 * @response: Pointer to fastboot response buffer
 *
 * Send a bare OKAY fastboot response. This is used where the command is
 * valid, but all the work is done after the response has been sent (e.g.
 * boot, reboot etc.)
 */
static void okay(char *cmd_parameter, char *response)
{
	fastboot_okay(NULL, response);
}

/**
 * unrecognized() - Send FAIL response
 *
 * @cmd_parameter: Pointer to command parameter
 * @response: Pointer to fastboot response buffer
 *
 * Send a FAIL fastboot response. This is used where the command is invalid
 */
static void unrecognized(char *cmd_parameter, char *response)
{
	fastboot_fail("unrecognized command", response);
}

/**
 * getvar() - Read a config/version variable
 *
 * @cmd_parameter: Pointer to command parameter
 * @response: Pointer to fastboot response buffer
 */
static void getvar(char *cmd_parameter, char *response)
{
#ifdef CONFIG_CMD_OTPCTRL
	unsigned long otp_cmd = 0x00;
	unsigned long dataSize = 1;
	char err_msg[MAX_ERR_MSG_LENGTH];
	if (!strncmp(OTP_PREFIX, cmd_parameter, strlen(OTP_PREFIX))) {
		char* cmd_param = cmd_parameter + strlen(OTP_PREFIX);
		if (!oem_otp_strcmp_funct(cmd_param, &otp_cmd, NULL, &dataSize, NULL))
		{
			snprintf(err_msg, MAX_ERR_MSG_LENGTH, "no cmd %s", cmd_param);
			fastboot_fail(err_msg, response);
			return;
		}
		oem_otp_r(otp_cmd, dataSize, response);
	}
	else
#endif
	{
		fastboot_getvar(cmd_parameter, response);
	}
}

/**
 * fastboot_download() - Start a download transfer from the client
 *
 * @cmd_parameter: Pointer to command parameter
 * @response: Pointer to fastboot response buffer
 */
static void download(char *cmd_parameter, char *response)
{
	char *tmp;

	if (!cmd_parameter) {
		fastboot_fail("Expected command parameter", response);
		return;
	}
	fastboot_bytes_received = 0;
	fastboot_bytes_expected = simple_strtoul(cmd_parameter, &tmp, 16);
	if (fastboot_bytes_expected == 0) {
		fastboot_fail("Expected nonzero image size", response);
		return;
	}

	/*
	 * Nothing to download yet. Response is of the form:
	 * [DATA|FAIL]$cmd_parameter
	 *
	 * where cmd_parameter is an 8 digit hexadecimal number
	 */

	if (fastboot_bytes_expected > fastboot_buf_size) {
		printf("%s failed, expected %d buf %d\n", __FUNCTION__,
			fastboot_bytes_expected, fastboot_buf_size);
		fastboot_fail(cmd_parameter, response);
	} else if (!is_safe_to_write(fastboot_buf_addr,
	           fastboot_bytes_expected)) {
		fastboot_fail(cmd_parameter, response);
	} else {
		printf("Starting download of %d bytes max %d\n",
		       fastboot_bytes_expected, fastboot_buf_size);
		fastboot_response("DATA", response, "%s", cmd_parameter);
	}
}

/**
 * fastboot_data_remaining() - return bytes remaining in current transfer
 *
 * Return: Number of bytes left in the current download
 */
u32 fastboot_data_remaining(void)
{
	return fastboot_bytes_expected - fastboot_bytes_received;
}

/**
 * fastboot_data_download() - Copy image data to fastboot_buf_addr.
 *
 * @fastboot_data: Pointer to received fastboot data
 * @fastboot_data_len: Length of received fastboot data
 * @response: Pointer to fastboot response buffer
 *
 * Copies image data from fastboot_data to fastboot_buf_addr. Writes to
 * response. fastboot_bytes_received is updated to indicate the number
 * of bytes that have been transferred.
 *
 * On completion sets image_size and ${filesize} to the total size of the
 * downloaded image.
 */
void fastboot_data_download(const void *fastboot_data,
			    unsigned int fastboot_data_len,
			    char *response)
{
#define BYTES_PER_DOT	0x20000
	u32 pre_dot_num, now_dot_num;

	if (fastboot_data_len == 0 ||
	    (fastboot_bytes_received + fastboot_data_len) >
	    fastboot_bytes_expected) {
		fastboot_fail("Received invalid data length",
			      response);
		return;
	}
	/*
	 * very unlikely to hit this if condition as stack overwrite
	 * is already checked at the beginning of the download.
	 * still worth checking between chunks.
	 */
	if (!is_safe_to_write(fastboot_buf_addr + fastboot_bytes_received,
	    fastboot_data_len)) {
		fastboot_fail("Insufficient memory for file download", response);
		return;
	}
	/* Download data to fastboot_buf_addr */
	memcpy(fastboot_buf_addr + fastboot_bytes_received,
	       fastboot_data, fastboot_data_len);

	pre_dot_num = fastboot_bytes_received / BYTES_PER_DOT;
	fastboot_bytes_received += fastboot_data_len;
	now_dot_num = fastboot_bytes_received / BYTES_PER_DOT;

	if (pre_dot_num != now_dot_num) {
		putc('.');
		if (!(now_dot_num % 74))
			putc('\n');
	}
	*response = '\0';
}

/**
 * fastboot_data_complete() - Mark current transfer complete
 *
 * @response: Pointer to fastboot response buffer
 *
 * Set image_size and ${filesize} to the total size of the downloaded image.
 */
void fastboot_data_complete(char *response)
{
#define FB_FILESZ 16
	char filesz[FB_FILESZ];
	/* Download complete. Respond with "OKAY" */
	fastboot_okay(NULL, response);
	printf("\ndownloading of %d bytes finished\n", fastboot_bytes_received);
	image_size = fastboot_bytes_received;
	snprintf(&filesz[0], FB_FILESZ, "%d", image_size);
	setenv("filesize", filesz);
	fastboot_bytes_expected = 0;
	fastboot_bytes_received = 0;
}

#ifdef CONFIG_CMD_OTPCTRL
extern void otp_ctrl_W(unsigned long  otp_cmd,unsigned long  val);
extern void otp_ctrl_R(unsigned long  otp_cmd);
extern void otp_show_results(unsigned long op, u8* dataArr, unsigned long dataSize);
#endif

#ifdef CONFIG_FASTBOOT_FLASH

#ifdef CONFIG_FASTBOOT_FLASH_NAND
/*
 * Set the UBI partition to search for the named volume, only "UBI" exists
 * at present but we may need to extend this API.
 */

static int mount_ubi_partition(char *cmd_parameter)
{
	/* TODO: What's the VID header all about? */
	if (ubi_part(cmd_parameter, (char *)NULL)) {
		printf("Cannot find UBI partition %s\n", cmd_parameter);
		return -1;
	}

	return 0;
}

/*
 * Extract the volume name from the part:vol string.
 */

#define UBIPREFIX "ubi#"
#define CISNAME "cis"
#define UNLOCKSIG "unlock"
#define FORCEPREFIX "force"
#ifdef CONFIG_CMD_OTPCTRL
#define RSAKEYNAME "otprsankey"
#define AESKEYNAME "otpaeskey"
// An unprogrammed AES key will return nonzero default data.
static const u8 AES_DEFAULT_DATA[] = {
	0x95, 0xc7, 0x04, 0x3e,
	0x08, 0x29, 0x0b, 0xdc,
	0xa4, 0x8e, 0x06, 0xbf,
	0xaf, 0xc2, 0x93, 0x45
};
#endif

static char *get_ubi_vol_name(char *cmd_parameter)
{
	char *rv;

	rv = cmd_parameter;
	rv += strlen(UBIPREFIX);

	return rv;
}

/*
 * determine if the partition name is CIS, meaning CIS partition table space
 */

static int is_CIS_partition(char *cmd_parameter)
{
	if (!strncmp(cmd_parameter,CISNAME, strlen(CISNAME))){
		printf("Detected CIS partition ID = %s\n", cmd_parameter);
		return 1;
	}
	return 0;
}

/*
 * determine if the partition name is actually an UBI volume that exists
 */

static int is_ubi_volume(char *cmd_parameter)
{
	if (strncmp(cmd_parameter,UBIPREFIX, strlen(UBIPREFIX))){
		return 0;
	}

	printf("Found %s\n", UBIPREFIX);

	if (mount_ubi_partition("UBI")) {
		printf("Can't find UBI partition\n");
		return 0;
	}

	return 1;
}

static int is_unlock_signature(char *cmd_parameter)
{
	if (!strncmp(cmd_parameter,UNLOCKSIG, strlen(UNLOCKSIG))){
		return 1;
	}
	return 0;
}

static int is_force_flash(char **cmd_parameter)
{
	if (strncmp(*cmd_parameter, FORCEPREFIX, sizeof(FORCEPREFIX) - 1) == 0) {
		if ((*cmd_parameter)[sizeof(FORCEPREFIX) - 1] == ':') {
			*cmd_parameter += sizeof(FORCEPREFIX);
			return 1;
		}
	}
	return 0;
}
/*
 * Uses the uBoot MStar writecis command to flash the CIS space.
 * Expects the first 512 bytes of the fastboot buffer to contain the PNI data
 * And the SPINANDINFO data to start at buffer + 0x200 bytes address
 */
static int write_cis(void)
{
	char * const argv[] = {"not_used","0x21000200","0x21000000"};

	printf("Calling writecis 0x%p 0x%p\n",fastboot_buf_addr + 0x200,
					fastboot_buf_addr);

	writeSpinandCIS(NULL, 1, 3, argv);
	return 0;
}

static int write_ubi_volume(char *cmd_parameter)
{
	char *vol_name = get_ubi_vol_name(cmd_parameter);
	int ret;

	/* write UBI volume, should have selected UBI partition already */
	ret = ubi_volume_write(vol_name, fastboot_buf_addr, image_size);

	printf("UBI volume %s write = %d\n", vol_name, ret);

	return ret;
}
#endif

#ifdef CONFIG_CMD_OTPCTRL
static int checkZero(u8* readback, int arrSize)
{
	int cnt = 0;
	for (int i = 0; i < arrSize; i++) {
		if (readback[i] == 0x00) {
			cnt++;
		}
	}
	return cnt;
}

static int write_chip_otp(u32 *data, int num_bytes, unsigned long otp_cmd, u8 *default_data, char *response)
{
	static u32 readback[MAX_KEY_DATA_SIZE/OTP_CELL_SIZE] = {0};
	const unsigned long op_w = 0x1;
	const unsigned long op_r = 0x2;
	char err_msg[MAX_ERR_MSG_LENGTH];
	int retry = 0;
	int ret = -1;

	if (num_bytes & (OTP_CELL_SIZE - 1)) {
		// Only lengths which are multiples of 4 are supported by this function
		fastboot_fail("bad flash len", response);
		return ret;
	}
	if (MAX_KEY_DATA_SIZE < num_bytes) {
		fastboot_fail("data size exceeds limitation", response);
		return ret;
	}
	do {
		otp_ctrl_R((otp_cmd << CMD_OFFSET) | 0x00);
		otp_show_results(op_r, (u8*)readback, num_bytes);
		if (0 == memcmp(data, readback, num_bytes)) {
			printf("otpw OK\n");
			fastboot_okay(NULL, response);
			ret = 0;
			break;
		} else {
			if (((default_data == NULL) && checkZero((u8*)readback, num_bytes) == num_bytes) ||
			    (default_data && (0 == memcmp(default_data, (u8*)readback, num_bytes)))) {
				retry++;
				if (retry > RETRY_THRESHOLD) {
					snprintf(err_msg, MAX_ERR_MSG_LENGTH, "otpw fail: readback %d", retry);
					printf("%s\n", err_msg);
					fastboot_fail(err_msg, response);
					ret = -1;
					break;
				}
			} else {
				snprintf(err_msg, MAX_ERR_MSG_LENGTH, "otpw fail: no match %d", retry);
				printf("%s\n", err_msg);
				fastboot_fail(err_msg, response);
				ret = -1;
				break;
			}
		}
		for (int i = 0; i < num_bytes/OTP_CELL_SIZE; i++) {
			otp_ctrl_W((otp_cmd << CMD_OFFSET) | OTP_CELL_SIZE*i, data[i]);
			otp_show_results(op_w, NULL, 0);
			ret = 0;
		}
	} while (retry <= RETRY_THRESHOLD);

	return ret;
}
#endif

/**
 * flash() - write the downloaded image to the indicated partition.
 *
 * @cmd_parameter: Pointer to partition name
 * @response: Pointer to fastboot response buffer
 *
 * Writes the previously downloaded image to the partition indicated by
 * cmd_parameter. Writes to response.
 */
static void flash(char *cmd_parameter, char *response)
{
	int rv;
#ifdef CONFIG_CMD_OTPCTRL
	unsigned long otp_cmd = 0x00;
	unsigned long data_size = 1;
	u8 *default_data = NULL;
#endif

#ifdef CONFIG_FASTBOOT_FLASH_NAND
	unlock_state_t unlock_state;

	if (is_unlock_signature(cmd_parameter)) {
		printf("Flashing unlock signature\n");
		rv = write_unlock_signature(fastboot_buf_addr, image_size);
		if (rv) {
			fastboot_fail("Writing unlock signature failed", response);
			return;
		} else {
			printf("Writing unlock signature succeeded\n");
			fastboot_okay(NULL, response);
			return;
		}
	}

	is_unlocked(&unlock_state);
	if (unlock_state < UNLOCK_STATE_RESTRICTED_UNLOCK) {
		printf("Flashing not allowed on locked device\n");
		fastboot_fail("Flashing not allowed!", response);
		return;
	}

	if (is_CIS_partition(cmd_parameter)) {
		printf("Flashing CIS partition table\n");
		if ((rv = write_cis())){
			printf("Write CIS failed %d\n", rv);
			fastboot_fail("Write CIS failed", response);
			return;
		} else {
			printf("Write CIS OK\n");
			fastboot_okay(NULL, response);
			return;
		}
	} else if (is_ubi_volume(cmd_parameter)) {
		printf("Flashing UBI NAND partition\n");
		if ((rv = write_ubi_volume(cmd_parameter))) {
			printf("Write UBI volume failed %d\n", rv);
			fastboot_fail("Write UBI failed", response);
			return;
		} else {
			printf("Write UBI volume succeeded\n");
			fastboot_okay(NULL, response);
			return;
		}
	} else if (is_otp_region(cmd_parameter)) {
		printf("Flashing NAND OTP region\n");
		if ((rv = write_nand_otp(fastboot_buf_addr, image_size))) {
			printf("Write NAND OTP failed %d\n", rv);
			fastboot_fail("Write OTP failed", response);
			return;
		} else {
			printf("Write NAND OTP succeeded\n");
			fastboot_okay(NULL, response);
			return;
		}
	}
#ifdef CONFIG_CMD_OTPCTRL
	else if (oem_otp_strcmp_funct(cmd_parameter, &otp_cmd, NULL, &data_size, &default_data)) {
		write_chip_otp((u32*)fastboot_buf_addr, data_size, otp_cmd, default_data, response);
	}
#endif
	else {
		if (is_force_flash(&cmd_parameter)) {
			fastboot_nand_flash_write(cmd_parameter,
			                          fastboot_buf_addr,
			                          image_size, response);
		} else {
			fastboot_nand_flash_write_if_modified(cmd_parameter,
			                                      fastboot_buf_addr,
			                                      image_size, response);
		}
	}
#endif
#ifdef CONFIG_FASTBOOT_FLASH_MMC
	fb_mmc_flash_write(cmd_parameter, fastboot_buf_addr, image_size,
				 response);
#endif
#if CONFIG_FASTBOOT_FLASH_CUS
	fastboot_cus_flash_write(cmd_parameter, fastboot_buf_addr, image_size,
				  response);
#endif
}

/**
 * erase() - erase the indicated partition.
 *
 * @cmd_parameter: Pointer to partition name
 * @response: Pointer to fastboot response buffer
 *
 * Erases the partition indicated by cmd_parameter (clear to 0x00s). Writes
 * to response.
 */
static void erase(char *cmd_parameter, char *response)
{
#ifdef CONFIG_FASTBOOT_FLASH_MMC
	fastboot_mmc_erase(cmd_parameter, response);
#endif
#ifdef CONFIG_FASTBOOT_FLASH_NAND
	fastboot_nand_erase(cmd_parameter, response);
#endif
#if CONFIG_FASTBOOT_FLASH_CUS
	fastboot_cus_erase(cmd_parameter, response);
#endif
}
#endif

/**
 * reboot_bootloader() - Sets reboot bootloader flag.
 *
 * @cmd_parameter: Pointer to command parameter
 * @response: Pointer to fastboot response buffer
 */
static void reboot_bootloader(char *cmd_parameter, char *response)
{
	if (fastboot_set_reboot_flag())
		fastboot_fail("Cannot set reboot flag", response);
	else
		fastboot_okay(NULL, response);
}

#ifdef CONFIG_FASTBOOT_CMD_OEM
/**
 * oem_coda() - Reset the MT7686 into CODA mode for flashing
 *
 * @cmd_parameter: Pointer to command parameter
 * @response: Pointer to fastboot response buffer
 * NB: MStar GPIO API is homegrown not the standard uBoot.
 */

/*
 * SAV connects to MTK GPIO12, selects MTK UART download if low at boot.
 * SAV connects to MTK !CE, low for MTK to start, high holds in reset.
 * Need to leave 20ms delay for MTK to actually reset.
 */

#define MT7686_PWR_CYCLE_DELAY_US 20000

static void oem_coda(char *cmd_parameter, char *response)
{
	/* Set PWR_ENB high to keep SAV526 powered */
	printf("Enabling blue power\n");
	MDrv_GPIO_Pad_Set(GPIO_PWR_ENB);
	mdrv_gpio_set_high(GPIO_PWR_ENB);

	/* Set GPIO low to enter CODA mode */
	MDrv_GPIO_Pad_Set(GPIO_CODA_MODE);
	mdrv_gpio_set_low(GPIO_CODA_MODE);

	/* Red WF->SAV UART pins as inputs */
	MDrv_GPIO_Pad_Set(GPIO_FUART_RX);
	mdrv_gpio_set_input(GPIO_FUART_RX);
	MDrv_GPIO_Pad_Set(GPIO_FUART_TX);
	mdrv_gpio_set_input(GPIO_FUART_TX);

	/* Toggle the RST_WF to high then low to reset the MT76XX */
	printf("Reset red\n");
	MDrv_GPIO_Pad_Set(GPIO_RST_WF);
	mdrv_gpio_set_high(GPIO_RST_WF);
	udelay(MT7686_PWR_CYCLE_DELAY_US);
	mdrv_gpio_set_low(GPIO_RST_WF);

	/* Apparently cannot fail!! */
	fastboot_okay(NULL, response);
}

/**
 * oem_fbtool() - Reset MT7933 or enter download mode for flashing
 *
 * @cmd_parameter: Pointer to command parameter
 * @response: Pointer to fastboot response buffer
 * NB: MStar GPIO API is homegrown not the standard uBoot.
 */

#define MT7933_PWR_CYCLE_DELAY_US 1000
#define MT7933_BOOTSTRAP_DELAY_US 32000

#if defined(CONFIG_ARCH_INFINITY6E)
static void oem_fbtool(char *cmd_parameter, char *response)
{
	/* Set bootstrap pins */
	MDrv_GPIO_Pad_Set(GPIO_WF_TX_IRQ);
	MDrv_GPIO_Pad_Set(GPIO_CTR_GPIO_B_2);
	MDrv_GPIO_Pad_Set(GPIO_CTR_GPIO_B_12);
	if (strcmp(cmd_parameter, "reset") == 0) {
		mdrv_gpio_set_high(GPIO_WF_TX_IRQ);
		mdrv_gpio_set_low(GPIO_CTR_GPIO_B_2);
		mdrv_gpio_set_high(GPIO_CTR_GPIO_B_12);
		goto RESET_RED;
	} else if (strcmp(cmd_parameter, "uart") == 0) {
		printf("Select UART download mode\n");
		mdrv_gpio_set_high(GPIO_WF_TX_IRQ);
		mdrv_gpio_set_low(GPIO_CTR_GPIO_B_2);
		mdrv_gpio_set_low(GPIO_CTR_GPIO_B_12);
	} else if (strcmp(cmd_parameter, "usb") == 0) {
		printf("Select USB download mode\n");
		mdrv_gpio_set_high(GPIO_WF_TX_IRQ);
		mdrv_gpio_set_high(GPIO_CTR_GPIO_B_2);
		mdrv_gpio_set_high(GPIO_CTR_GPIO_B_12);
	} else {
		fastboot_fail("Unrecognized oem fbtool parameter\n", response);
		return;
	}

	/* Set PWR_ENB high to keep SAV533 powered */
	printf("Enabling blue power\n");
	MDrv_GPIO_Pad_Set(GPIO_PWR_ENB);
	mdrv_gpio_set_high(GPIO_PWR_ENB);

	/* Set Red WF -> Blue UART pins as inputs */
	if (strcmp(cmd_parameter, "uart") == 0) {
		MDrv_GPIO_Pad_Set(GPIO_FUART_RX);
		MDrv_GPIO_Pad_Set(GPIO_FUART_TX);
		mdrv_gpio_set_input(GPIO_FUART_RX);
		mdrv_gpio_set_input(GPIO_FUART_TX);
	}

RESET_RED:

	/* Toggle the RST_WF to high then low to reset the MT7933 */
	printf("Reset red\n");
	MDrv_GPIO_Pad_Set(GPIO_RST_WF);
	mdrv_gpio_set_high(GPIO_RST_WF);
	udelay(MT7933_PWR_CYCLE_DELAY_US);
	mdrv_gpio_set_low(GPIO_RST_WF);

	/* Unset bootstrap pins */
	udelay(MT7933_BOOTSTRAP_DELAY_US);
	mdrv_gpio_set_high(GPIO_WF_TX_IRQ);
	mdrv_gpio_set_low(GPIO_CTR_GPIO_B_2);
	mdrv_gpio_set_high(GPIO_CTR_GPIO_B_12);

	/* Presumably succeeded */
	fastboot_okay(NULL, response);
}
#else
static void oem_fbtool(char *cmd_parameter, char *response)
{
	fastboot_fail("Unsupported command\n", response);
}
#endif

/**
 * Set the UBI partition (mount MTD partition as UBI)
 *
 * @cmd_parameter: Pointer to command parameter
 * @response: Pointer to fastboot response buffer
 */
#define MAX_CMD_BUF 32
static void set_ubi_partition(char *cmd_parameter, char *response)
{
	char cmdbuf[MAX_CMD_BUF], *arg1;

	strtok(cmd_parameter, " ");
	arg1 = strtok(NULL," ");

	printf("%s input %s arg %s\n", __FUNCTION__, cmd_parameter, arg1);

	snprintf(cmdbuf, MAX_CMD_BUF, "ubi part %s", arg1);
	printf(cmdbuf);
	if (run_command(cmdbuf, 0))
		fastboot_fail("", response);
	else
		fastboot_okay(NULL, response);
}

/**
 * create_ubi_vol() - Create an empty UBI volume
 *
 * @cmd_parameter: Pointer to command parameter
 * @response: Pointer to fastboot response buffer
 */

static void create_ubi_vol(char *cmd_parameter, char *response)
{
	char cmdbuf[MAX_CMD_BUF], *name, *size;

	printf("%s arg %s\n", __FUNCTION__, cmd_parameter);

	strtok(cmd_parameter, " ");
	name = strtok(NULL, " ");
	size = strtok(NULL, " ");

	printf("name %s size %s\n", name, size);

	snprintf(cmdbuf, MAX_CMD_BUF, "ubi create %s %s", name, size);

	printf("command %s\n", cmdbuf);
	if (run_command(cmdbuf, 0))
		fastboot_fail("", response);
	else
		fastboot_okay(NULL, response);
}

static void remove_ubi_vol(char *cmd_parameter, char *response)
{
	char cmdbuf[MAX_CMD_BUF], *volname;

	printf("%s arg %s\n", __FUNCTION__, cmd_parameter);

	strtok(cmd_parameter, " ");
	volname = strtok(NULL, " ");

	snprintf(cmdbuf, MAX_CMD_BUF, "ubi remove %s", volname);
	if (run_command(cmdbuf, 0))
		fastboot_fail("", response);
	else
		fastboot_okay(NULL, response);
}

static void run(char *cmd_parameter, char *response)
{
	char cmdbuf[MAX_CMD_BUF];

	snprintf(cmdbuf, MAX_CMD_BUF, "%s", cmd_parameter);
	printf("Running cmdline: %s\n",cmdbuf);
	if (run_command(cmdbuf, 0))
		fastboot_fail("", response);
	else
		fastboot_okay(NULL, response);
}

/*
 * To allow working from empty flash (and making one)
 * 1) erase UBI partition "oem ubi clear partition_name"
 * 2) ubi part <partition_name> "oem ubi mount partition_name"
 * 	NB! Something needs to allow ubi part UBI to really mount UBI partition.
 *	NB! New UBI command, like "ubi decache"
 * 3) create volume on partition "oem ubi cvol volume_name size"
 */

static void oem_ubi(char *cmd_parameter, char *response)
{
	/* Parse UBI commands from cmd_parameter string */
	if (!strncmp(cmd_parameter,"setpart",strlen("setpart")))
		set_ubi_partition(cmd_parameter, response);
	else if (!strncmp(cmd_parameter,"mkvol",strlen("cvol")))
		create_ubi_vol(cmd_parameter, response);
	else if (!strncmp(cmd_parameter,"rmvol",strlen("evol")))
		remove_ubi_vol(cmd_parameter, response);
	else if (!strncmp(cmd_parameter,"runcmd",strlen("runcmd")))
		run(cmd_parameter, response);
	else
		fastboot_fail("Unrecognised UBI oem command", response);
}

static void oem_setserial(char *cmd_parameter, char *response)
{
	if (setenv("serial#", cmd_parameter)) {
		printf("Failed to set serial# to %s\n", cmd_parameter);
		fastboot_fail("Failed to set serial#", response);
	} else {
		saveenv();
		fastboot_okay(NULL, response);
	}
}

static void oem_setvar(char *cmd_parameter, char *response)
{
	int i=0;
	int paramlen = strlen(cmd_parameter);
	int varlen;
	char* varval;

	for(i = 0; i < paramlen; i++) {
		if ((cmd_parameter[i] == ':') || (cmd_parameter[i] == ' ')) {
			cmd_parameter[i] = '\0';
			/* Allow variables with ' ' and ':'' in them' */
			break;
		}
	}

	varlen = strlen(cmd_parameter);
	if (varlen == paramlen) {
		fastboot_fail("Expected variable name and value", response);
		return;
	}

	varval = cmd_parameter + varlen + 1;

	if (setenv(cmd_parameter, varval)) {
		printf("Failed to set %s to %s\n", cmd_parameter, varval);
		fastboot_fail("Failed to set environment variable", response);
	} else {
		saveenv();
		fastboot_okay(NULL, response);
	}
}

extern int relock_device(void);
#ifdef CONFIG_FASTBOOT_FLASH_NAND
static void oem_otp_nand_lock(char *cmd_parameter, char *response)
{
	int ret = fb_otp_nand_lock();
	if (ret == 0) {
		fastboot_okay(NULL, response);
		/* Ensure the device is set as locked */
		relock_device();
	} else {
		fastboot_fail("Failed to lock NAND OTP", response);
	}
}

extern int validate_ubi_partitions(void);
static void oem_ubi_validate(char *cmd_parameter, char *response)
{
	int ret = validate_ubi_partitions();
	if (ret == 0) {
		fastboot_okay(NULL, response);
	} else {
		fastboot_fail("UBI validation failed", response);
	}
}

#endif

// for all the commands, please refer to the Jira - COCOA-12276 for further
// detail https://issues.labcollab.net/browse/COCOA-12276

#ifdef CONFIG_CMD_OTPCTRL
static bool oem_otp_strcmp_funct(char *cmd_parameter,
                                 unsigned long* otp_cmd,
                                 unsigned long* val,
                                 unsigned long* dataSize,
                                 u8** defaultData)
{
	if (!otp_cmd || !dataSize) {
		return FALSE;
	}
	if (!strcmp(cmd_parameter, "keylock")) {
		/* otpw: lock both E key and N key from being programmed */
		/* otpr: read keylock bit */
		*otp_cmd = 0x03;
		*dataSize = 1;
		if (val) {
			*val = 0x01;
		}
	} else if (!strcmp(cmd_parameter, "secureboot")) {
		/* otpw: enable secure boot */
		/* otpr: read secure boot bit */
		*otp_cmd = 0x02;
		*dataSize = 1;
		if (val) {
			*val = 0x01;
		}
	} else if (!strcmp(cmd_parameter, "debugclose")) {
		/* otpw: disable JTAG debug */
		/* otpr: read disable JTAG bit */
		*otp_cmd = 0x10;
		*dataSize = 1;
		if (val) {
			*val = 0x01;
		}
	} else if (!strcmp(cmd_parameter, "cid")) {
		/* otpw: program CID */
		/* otpr: read cid byte */
		*otp_cmd = 0x09;
		*dataSize = 1;
		if (val) {
			*val = CID;
		}
	} else if (!strcmp(cmd_parameter, "cidlock")) {
		/* otpw: lock CID from being programmed */
		/* otpr: read CID lock bit */
		*otp_cmd = 0x1a;
		*dataSize = 1;
		if (val) {
			*val = 0x01;
		}
	} else if (!strcmp(cmd_parameter, "flashspeed")) {
		/* otpw: enable flash speedup */
		/* otpr: read flash speedup bit */
		*otp_cmd = 0x18;
		*dataSize = 1;
		if (val) {
			*val = 0x01;
		}
	} else if (!strcmp(cmd_parameter, "aesdma")) {
		/* otpw: enter AESDMA speed up mode */
		/* otpr: read AESDMA speed up bit */
		*otp_cmd = 0x19;
		*dataSize = 1;
		if (val) {
			*val = 0x01;
		}
	} else if (!strcmp(cmd_parameter, "enablebootsrc")) {
		/* otpw: enable bootsrc select from OTP */
		/* otpr: read enable bootsrc bit */
		*otp_cmd = 0x1b;
		*dataSize = 1;
		if (val) {
			*val = 0x01;
		}
	} else if (!strcmp(cmd_parameter, "bootsrc")) {
		/* otpw: bootsrc selct */
		/* otpr: read bootsrc select bit*/
		*otp_cmd = 0x1c;
		*dataSize = 1;
		if (val) {
			*val = BOOT_SRC;
		}
	} else if (!strcmp(cmd_parameter, "bootsrclock")) {
		/* otpw: lock bootsrc from being programmed */
		/* otpr: read bootsrc lock bit */
		*otp_cmd = 0x1d;
		*dataSize = 1;
		if (val) {
			*val = 0x01;
		}
	} else if (!strcmp(cmd_parameter, "antirollback")) {
		/* otpw: enable AntiRollBack check */
		/* otpr: read AntiRollBack bit */
		*otp_cmd = 0x12;
		*dataSize = 1;
		if (val) {
			*val = 0x01;
		}
	} else if (!strcmp(cmd_parameter, "ekey")) {
		/* otpw: program E-key */
		/* otpr: read ekey */
		*otp_cmd = 0x01;
		*dataSize = E_KEY_SIZE;
		if (val) {
			*val = E_KEY;
		}
	} else if (!strcmp(cmd_parameter, "aeskeylock")) {
		/* otpw: lock aes key from being programmed */
		/* otpr: read aes key lock bit */
		*otp_cmd = 0x06;
		*dataSize = 1;
		if (val) {
			*val = 0x1;
		}
	} else if (!strcmp(cmd_parameter, "aeskeyblock")) {
		/* otpw: block aes key from being read by SW */
		/* otpr: read aes key block bit */
		*otp_cmd = 0x06;
		*dataSize = 1;
		if (val) {
			*val = 0x4;
		}
	// These are programmed using fastboot flash
	} else if (!strncmp(cmd_parameter, RSAKEYNAME, strlen(RSAKEYNAME))) {
		*otp_cmd = 0x00;
		if (defaultData) {
			*defaultData = NULL;
		}
		*dataSize = RSA_KEY_SIZE;
	} else if (!strncmp(cmd_parameter, AESKEYNAME, strlen(AESKEYNAME))) {
		*otp_cmd = 0x04;
		if (defaultData) {
			*defaultData = AES_DEFAULT_DATA;
		}
		*dataSize = AES_KEY_SIZE;
	} else {
		return FALSE;
	}
	return TRUE;
}

static void oem_otp_w(char *cmd_parameter, char *response)
{
	unsigned long otp_cmd = 0x00;
	unsigned long val = 0x00;
	unsigned long dataSize = 1;
	const unsigned long op_w = 0x1;
	const unsigned long op_r = 0x2;
	char err_msg[MAX_ERR_MSG_LENGTH];
	int retry = 0;
	u32 data_4bytes = 0;
	if (!oem_otp_strcmp_funct(cmd_parameter, &otp_cmd, &val, &dataSize, NULL)) {
		snprintf(err_msg, MAX_ERR_MSG_LENGTH, " %s: no cmd", cmd_parameter);
		fastboot_fail(err_msg, response);
		return;
	}
	if (dataSize > 4) {
		/* anything over 4 bytes need to use fastboot flash */
		snprintf(err_msg, MAX_ERR_MSG_LENGTH, "%s: data too long", cmd_parameter);
		fastboot_fail(err_msg, response);
	}
	do {
		otp_ctrl_R((otp_cmd << CMD_OFFSET) | 0x00); // read back first to make sure default is 0x00
		otp_show_results(op_r, (u8*)&data_4bytes, dataSize);
		if (data_4bytes) {
			if (!strcmp(cmd_parameter, "bootsrc") || !strcmp(cmd_parameter, "cid") || !strcmp(cmd_parameter, "ekey")) {
				if (data_4bytes != val) {
					snprintf(err_msg, MAX_ERR_MSG_LENGTH, "%s: no match", cmd_parameter);
					fastboot_fail(err_msg, response);
					break;
				}
			}
			fastboot_okay(NULL, response);
			break;
		} else {
			retry++;
			if (retry > RETRY_THRESHOLD) {
				snprintf(err_msg, MAX_ERR_MSG_LENGTH, "%s: failed", cmd_parameter);
				fastboot_fail(err_msg, response);
				break;
			}
		}
		otp_ctrl_W((otp_cmd << CMD_OFFSET) | 0x00, val);
		otp_show_results(op_w, NULL, 0);
	} while(retry <= RETRY_THRESHOLD);
}
static void oem_otp_r(unsigned long otp_cmd, unsigned long dataSize, char *response)
{
	const unsigned long op_r = 0x2;
	char data_msg[MAX_DATA_MSG_LENGTH];
	u32 data_4bytes = 0;
	if (dataSize > sizeof(data_4bytes)) {
		fastboot_fail("size too big", response);
		return;
	}
	otp_ctrl_R((otp_cmd << CMD_OFFSET) | 0x00);
	otp_show_results(op_r, (u8*)&data_4bytes, dataSize);
	snprintf(data_msg, MAX_DATA_MSG_LENGTH, "%8x", data_4bytes);
	fastboot_okay(data_msg, response);
}
#endif

static void oem_setfactory(char *cmd_parameter, char *response)
{
	int paramlen = strlen(cmd_parameter);
	char *tmp;

	uint32_t port = 0;
	if (paramlen > 0) {
		port = simple_strtoul(cmd_parameter, &tmp, 10);
	}
	if (port) {
		if (enable_factory_test_mode(port) < 0)
			fastboot_fail("Failed to enable factory test mode", response);
		else {
			printf("Set Factory Test Mode with port %d\n", port);
			fastboot_okay(NULL, response);
		}
	} else {
		disable_factory_test_mode();
		printf("Disabled Factory Test Mode\n");
		fastboot_okay(NULL, response);
	}
}
#endif

static void oem_relock(char *cmd_parameter, char *response)
{
	int ret = relock_device();
	if (ret == 0) {
		fastboot_okay(NULL, response);
	} else {
		fastboot_fail("Failed to relock device", response);
	}
}
